#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "ipv46.h"

#define PE(s...)	fprintf(stderr, s)
#define EPE(s...)	fprintf(stderr, s, strerror(errno))

// ***********************************************************************
// Defaults
// ***********************************************************************
#define VERB_NONE	0
#define VERB_NORMAL	1
#define VERB_DUMP	2
int verbosity = VERB_NORMAL;
#define MODE_NONE	0
#define MODE_SERVER	1
#define MODE_CLIENT	2
static int mode = MODE_NONE;
#define DEFAULT_IPVER	4
static int ip_version = DEFAULT_IPVER;
#define DEFAULT_BUFFSZ	131072
static int buffer_size = DEFAULT_BUFFSZ;
#define DEFAULT_PORT	19000
#define DUMP_RATE	25

// **********************************************************
// Checksums
// **********************************************************
static unsigned int checksum(void *data, unsigned int size)
{
	unsigned int t, sz, v = 0;
	
	sz = size>>2;
	for (t=0; t<sz; t++) {
		v ^= ((unsigned int *)data)[t] + t;
	}
	for (t=size&0xfffffffc; t<size; t++) {
		v ^= ((unsigned char *)data)[t] + t;
	}
	return(v ^ 0x80ffa581);
}

// **********************************************************
// Effective transfer
// **********************************************************
static int do_transfer(int sfd, int infd, int outfd)
{
	fd_set rfd;
	void *m = NULL;
	int tin=0, tout=0, incs=0, outcs=0;
	int cps, beg, now, last, dly;
	int n, i, r;
	
	m = malloc(buffer_size);
	if (!m) {
		PE("Can't allocate %dkB for transfer\n", buffer_size>>10);
		return(1);
	}
	
	cps = sysconf(_SC_CLK_TCK);
	beg = times(NULL);
	dly = cps / DUMP_RATE;
	if (dly < 1) dly = 1;
	last = beg - dly;
	
	while(1) {
		now = times(NULL);
		if ((now>=last+dly)&&(verbosity==VERB_DUMP)) {
			if (now==beg) now = beg+1;
			PE("Sent: %dkB (%dkB/s)     "
				"Received: %dkB (%dkB/s)     \r",
				tout>>10, (tout>>10)*cps/(now-beg),
				tin>>10, (tin>>10)*cps/(now-beg));
			last = now;
		}
		FD_ZERO(&rfd);
		FD_SET(infd, &rfd);
		FD_SET(sfd, &rfd);
		if (select(sfd>infd?sfd+1:infd+1, &rfd, NULL, NULL, NULL)<0) {
			if (verbosity==VERB_DUMP) PE("\n");
			EPE("Select error: %s\n");
			close(sfd);
			return(1);
		}

		// local input -> socket
		if (FD_ISSET(infd, &rfd)) {
			n = read(infd, m, buffer_size);
			if (n < 0) {
				if (verbosity==VERB_DUMP) PE("\n");
				EPE("Read error: %s\n");
				close(sfd);
				return(1);
			}
			if (n == 0) {
				if (verbosity==VERB_DUMP) PE("\n");
				if (verbosity) PE("End of file\n"
					" Sent     %9d bytes, checksum: 0x%08x\n"
					" Received %9d bytes, checksum: 0x%08x\n",
						tout, outcs, tin, incs);
				close(sfd);
				return(0);
			}
			outcs += checksum(m, n) ^ tout;

			i = 0;
			while (i < n) {
				r = send(sfd, m+i, n-i, 0);
				if (r <= 0) {
					if (verbosity==VERB_DUMP) PE("\n");
					EPE("Send error: %s\n");
					close(sfd);
					return(1);
				}
				i += r;
			}
			tout += n;
		}

		// socket -> local output
		if (FD_ISSET(sfd, &rfd)) {
			n = recv(sfd, m, buffer_size, 0);
			if (n < 0) {
				if (verbosity==VERB_DUMP) PE("\n");
				EPE("Receive error: %s\n");
				close(sfd);
				return(1);
			}
			if (n == 0) {
				if (verbosity==VERB_DUMP) PE("\n");
				if (verbosity) PE("Connection closed\n"
					" Received %9d bytes, checksum: 0x%08x\n"
					" Sent     %9d bytes, checksum: 0x%08x\n",
						tin, incs, tout, outcs);
				close(sfd);
				exit(0);
			}
			incs += checksum(m, n) ^ tin;
			i = 0;
			while (i < n) {
				r = write(outfd, m+i, n-i);
				if (r <= 0) {
					if (verbosity==VERB_DUMP) PE("\n");
					EPE("Write error: %s\n");
					close(sfd);
					return(1);
				}
				i += r;
			}
			tin += n;
		}
	}
}

// **********************************************************
// Server-side
// **********************************************************
static int accept_loop(int sfd, int ipv, struct ip_addr *peer)
{
	char hstr[40];
	struct ip_addr c;
	int pfd;

	ip_setipv(&c, ipv);
	pfd = accept(sfd, &c.addr.gen, &c.addrsize);
	if (pfd < 0) {
		EPE("accept() error: %s\n");
		return(-1);
	}
	if (!ip_consistent(&c)) {
		if (verbosity == VERB_NORMAL) PE("X ");
		else if (verbosity == VERB_DUMP)
			PE("Connection with wrong protocol refused\n");
		close(pfd);
		return(-2);
	}
	if (peer) {
		if (!ip_sameaddr(&c, peer)) {
			if (verbosity == VERB_NORMAL) PE("X ");
			else if (verbosity == VERB_DUMP) {
				ip_addrtostr(hstr, &c);
				PE("Connection from %s (port %d) refused\n",
					hstr, ip_getport(&c));
			}
			close(pfd);
			return(-2);
		}
	}
	if (verbosity) {
		ip_addrtostr(hstr, &c);
		PE("OK (%s, port %d)\n", hstr, ip_getport(&c));
	}
	return(pfd);
}

// **********************************************************
// "Hand shaking"
// **********************************************************
int transf_stream(int isserver, struct ip_addr *server,
	struct ip_addr *peer, int infd, int outfd)
{
	int sfd, pfd;


	sfd = socket(server->pf, SOCK_STREAM, 0);
	if (sfd < 0) {
		EPE("Can't create a stream socket: %s\n");
		return(1);
	}
	
	if (!isserver) {
		if (verbosity) PE("Connecting... ");
		if (connect(sfd, &server->addr.gen, server->addrsize)) {
			if (verbosity) EPE("failed: %s\n");
			else EPE("Can't connect to server: %s\n");
			close(sfd);
			return(1);
		}
		if (verbosity) PE("OK\n");
		return(do_transfer(sfd, infd, outfd));
	}
	
	if (bind(sfd, &server->addr.gen, server->addrsize)) {
		EPE("Can't bind socket: %s\n");
		close(sfd);
		return(1);
	}
	if (listen(sfd, 256)) {
		EPE("Can't put socket in listen mode: %s\n");
		close(sfd);
		return(1);
	}
	if (verbosity == VERB_NORMAL) PE("Waiting... ");
	else if (verbosity == VERB_DUMP) PE("Server started...\n");
	
	do {
		pfd = accept_loop(sfd, server->ipv, peer);
		if (pfd == -1) { close(sfd); return(1); }
	} while(pfd<0);
	close(sfd);
	return(do_transfer(pfd, infd, outfd));
}

// ***********************************************************************
// Help
// ***********************************************************************
void dumphelp()
{
	printf(
	"Usage: nettrans {-waitas|-reach} <server address> [options]\n"
	"\t-waitas           Start a server and wait for the peer\n"
	"\t-reach            Connect to the server and run as client\n"
	"\t<server address>  Host address of the server\n"
	"Options:\n"
	"\t-port=<x>         Port of the server (default %d)\n"
	"\t-peer=<x>         (server) Only accept client with address <x>\n"
	"\t-verbose          Dump progress info to stderr\n"
	"\t-silent           Don't dump anything\n"
	"\t-in=<file>        Use <file> as input instead of stdin\n"
	"\t-out=<file>       Use <file> as output instead of stdout\n"
	"\t-buffer=<x>       Size of the buffer in kilobyte (default %dkB)\n"
	"\t-ipv4             Use IPv4 protocol%s\n"
	"\t-ipv6             Use IPv6 protocol%s\n",
	DEFAULT_PORT, DEFAULT_BUFFSZ>>10,
	(DEFAULT_IPVER==4)?" (default)":"",
	(DEFAULT_IPVER==6)?" (default)":"");
}


// ***********************************************************************
// Main
// ***********************************************************************
int main(int argc, char **argv)
{
	char astr[IP_ADDRESS_STRING_SIZE];
	char *server = NULL, *peer = NULL, *inf = NULL, *outf = NULL;
	struct ip_addr server_addr;
	struct ip_addr peer_addr;
	int input_fd, output_fd;
	char *e;
	int t, p=-1;

	if (argc < 2) { dumphelp(); return(0); }

	for (t=1; t<argc; t++) {
		if (!strcmp(argv[t], "--help")) {
			dumphelp();
			return(0);
		} else if (!strcmp(argv[t], "-waitas")) {
			if (mode!=MODE_NONE) {
				PE("'-waitas': mode already specified\n");
				return(1);
			}
			mode = MODE_SERVER;
		} else if (!strcmp(argv[t], "-reach")) {
			if (mode!=MODE_NONE) {
				PE("'-reach': mode already specified\n");
				return(1);
			}
			mode = MODE_CLIENT;
		} else if (!strncmp(argv[t], "-port=",6)) {
			if (p != -1) {
				PE("Port specified twice\n");
				return(1);
			}
			p = strtol(argv[t]+6,&e,0);
			if ((*e!=0)||(e==argv[t]+6)) {
				PE("Syntax error after '-port'\n");
				return(1);
			}
			if ((p<0)||(p>=65536)) {
				PE("Port out of range [0,65535]\n");
				return(1);
			}
		} else if (!strncmp(argv[t], "-peer=",6)) {
			if (peer) {
				PE("Peer specified twice\n");
				return(1);
			}
			peer = argv[t]+6;
		} else if (!strcmp(argv[t], "-verbose")) verbosity=VERB_DUMP;
		else if (!strcmp(argv[t], "-silent")) verbosity=VERB_NONE;
		else if (!strncmp(argv[t], "-in=",4)) {
			if (inf) {
				PE("Input file specified twice\n");
				return(1);
			}
			inf = argv[t]+4;
		} else if (!strncmp(argv[t], "-out=",5)) {
			if (outf) {
				PE("Output file specified twice\n");
				return(1);
			}
			outf = argv[t]+5;
		} else if (!strncmp(argv[t], "-buffer=",8)) {
			if (buffer_size > 0) {
				PE("Packet size specified twice\n");
				return(1);
			}
			buffer_size = strtol(argv[t]+8,&e,0)<<10;
			if ((*e!=0)||(e==argv[t]+8)||(buffer_size<=0)) {
				PE("Syntax error after '-buffer'\n");
				return(1);
			}
		} else if (!strcmp(argv[t], "-ipv4")) ip_version = 4;
		else if (!strcmp(argv[t], "-ipv6")) ip_version = 6;
		else {
			if (server) {
				PE("Unknown option (%s, %s)\n",
					server, argv[t]);
				return(1);
			}
			server = argv[t];
		}
	}
	
	if (mode==MODE_NONE) {
		PE("Mode of operation not specified\n");
		return(1);
	}
	if (!server) {
		PE("Server not specified\n");
		return(1);
	}
	if (p == -1) p = DEFAULT_PORT;
	
	// Peer
	if (peer) {
		if (mode!=MODE_SERVER) {
			PE("'-peer' option meaningless in client mode\n");
			return(1);
		}
		ip_setipv(&peer_addr, ip_version);
		if (verbosity) PE("Resolving '%s'... ", peer);
		if (ip_resolve(&peer_addr, peer)) {
			if (verbosity) PE("host not found\n");
			else PE("'%s': host not found\n", peer);
			return(1);
		}
		ip_addrtostr(astr, &peer_addr);
		if (verbosity) PE("%s\n", astr);
	}
	
	// Server
	ip_setipv(&server_addr, ip_version);
	if (verbosity) PE("Resolving '%s'... ", server);
	if (ip_resolve(&server_addr, server)) {
		if (verbosity) PE("host not found\n");
		else PE("'%s': host not found\n", server);
		return(1);
	}
	ip_addrtostr(astr, &server_addr);
	if (verbosity) PE("%s\n", astr);
	ip_setport(&server_addr, p);
	
	// Local files
	if (inf) {
		input_fd = open(inf, O_RDONLY);
		if (input_fd < 0) {
			PE("'%s': %s\n", inf, strerror(errno));
			return(1);
		}
	} else input_fd = 0;
	if (outf) {
		output_fd = open(outf, O_CREAT|O_WRONLY|O_EXCL,
			S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
		if (output_fd < 0) {
			PE("'%s': %s\n", outf, strerror(errno));
			return(1);
		}
	} else output_fd = 1;
	
	// Go
	return(transf_stream(mode!=MODE_CLIENT, &server_addr,
			peer?&peer_addr:NULL, input_fd, output_fd));
}

