#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include "YGL2/display.h"
#include "gui/images.h"
#include "timer.h"
#include "socket_utils.h"
#include "video_capture.h"
#include "video_main.h"

#define PO(s...)	printf(s)
#define PE(s...)	fprintf(stderr, s)
#define DEFAULT_PORT	18060
#define PLAY_PROGRAM	"/usr/bin/play"
#define MOVE_THRESHOLD	150

static char **global_envp;
static unsigned int last_move;
static int alert_level;

static struct {
	unsigned int delay;
	int inuse, usecount, maxuse;
	char *sounds[20];
} alert_levels[] =
{
	{ 10000, 0, 0, 20, {"/usr/share/sounds/shutdown1.wav", "s1.ogg", "s2.wav", NULL} },
	{ 20000, 0, 0, 5, {"/usr/share/sounds/Kopete_User_is_Online.ogg", "s4.wav", NULL } },
	{ 30000, 0, 0, 5, {"s5.ogg", NULL} },
	{ 0x7fffffff, 0, 0, 1, {"dummy", NULL} }
};
// ***********************************************************
// Alert
// ***********************************************************
void run_alert(int l)
{
	char *fn;
	char *argv[2];
	pid_t c;

	alert_levels[l].usecount++;
	if (alert_levels[l].usecount > alert_levels[l].maxuse) {
		alert_levels[l].inuse++;
		if (!alert_levels[l].sounds[alert_levels[l].inuse])
			alert_levels[l].inuse = 0;
	}
	fn = alert_levels[l].sounds[alert_levels[l].inuse];
	// ---------
	c = fork();
	if (c < -1) {
		PE("Can't fork! (%s)\n", strerror(errno));
		exit(1);
	}
	if (c != 0) return;
	argv[0] = PLAY_PROGRAM;
	argv[1] = fn;
	argv[2] = NULL;
	execve(argv[0], argv, global_envp);
	PE("!!! execve() error: %s\n", strerror(errno));
}

// ***********************************************************
// Window
// ***********************************************************
Window win_open()
{
	unsigned long attrmask;
	XSetWindowAttributes attr;
	Window vw;
	
	attrmask = CWBackPixmap|CWBackingStore|CWSaveUnder|CWEventMask;
	attr.background_pixmap = None;
	attr.backing_store = NotUseful;
	attr.save_under = False;
	attr.event_mask = KeyPressMask|ButtonPressMask|ButtonMotionMask
		|ButtonReleaseMask|ExposureMask|StructureNotifyMask;

	vw = XCreateWindow(ygl.display, ygl.root_window, 0, 0, 360, 240, 0,
		ygl.depth, InputOutput, ygl.default_visual, attrmask, &attr);
	if (vw == None) { PE("Can't create a window\n"); return(None); }
	YSetName(vw, "video capture driver for gembla", "video");
	XMapRaised(ygl.display, vw);
	
	XSync(ygl.display, False);
	return(vw);
}

// ***********************************************************
// Grab a frame
// ***********************************************************
static struct gui_ws ws;
static void *buf[2];
static int nextbuf;
void grab_new_frame(Window w)
{
	unsigned char *src, *cmp;
	unsigned int *dst;
	int c, c2, t, cnt;
	int backbuf;
	time_t curtime;

	backbuf = !nextbuf;
	memcpy(buf[nextbuf], video_map.data, video_map.size);
	src = (unsigned char *)buf[nextbuf];
	cmp = (unsigned char *)buf[backbuf];
	dst = (unsigned int *)ws.data;
	t = video_map.nbpix >> 1;
	cnt = 0;
	while (t>0) {
		c = (src[0] + src[2]) >> 1;
		c2 = (cmp[0] + cmp[2]) >> 1;
		c2 = c2-c;
		c2 = c2*c2;
		if (c2 > 1000) {
			*dst = 0x00ff00;
			cnt++;
		} else *dst = (c<<16)|(c<<8)|c;
		src += 4;
		cmp += 4;
		dst++;
		t--;
	}
	nextbuf = backbuf;

	gui_put_workspace(&ws, w, 0,0,0,0,360,240);
	XSync(ygl.display, False);
	if (cnt > MOVE_THRESHOLD) {
		PE("(%8d) MOVE MOVE MOVE\n", cnt);
		last_move = timer_ms();
		alert_level = -1;
	}
	curtime = time(NULL);
	PO("%8d\t%8d\t%s", timer_ms(), cnt, ctime(&curtime));
}

// ***********************************************************
// Main
// ***********************************************************
int main(int argc, char **argv, char **envp)
{
	Window w;
	int fd, cmd, n, incer;
	unsigned int delay;
	char *e;

	global_envp = envp;
	setlinebuf(stdout);

	if ((argc < 2) || (argc > 3)) {
		PE("Usage: video-driver <video_socket|-> [port]\n");
		return(argc > 1);
	}
	if (argc >= 3) {
		n = strtol(argv[2], &e, 0);
		if ((e==argv[2])||(*e!=0)||(n<0)||(n>=65536)) {
			PE("Port should be [0;65535]\n");
			return(1);
		}
	} else n = DEFAULT_PORT;
	
	if (capture_init()) return(1);
	buf[0] = malloc(video_map.size * 2);
	if (!buf[0]) return(1);
	buf[1] = buf[0] + video_map.size;
	nextbuf = 0;
	if (YInit()) return(1);
	if (ygl.bits_per_pixel != 32) {
		PE("Only 32bpp display supported\n");
		return(1);
	}
	if (gui_create_workspace(&ws, GUI_SHMPIXMAP|GUI_SHMIMAGE|GUI_IMAGE,
		360, 240)) return(1);
	w = win_open();
	if (w==None) return(1);
	
	if (!strcmp(argv[1], "-")) fd = 0;
	else {
		fd = socket_create_accept(argv[1], n);
		if (fd < 0) return(1);
	}
	PE("video-driver: connected\n");
	timer_init();
	last_move = timer_ms();
	alert_level = -1;
	alarm_ms(40, &incer, 1);
	incer = 0;
	do {
		cmd = 0;
		n = read(fd, &cmd, 1);
		if ((n <= 0)&&(errno==EINTR)) {
			n = 1;
		} else if (n == 1) {
			if (cmd == 2) n = 0;
			else {
				// cmd = XXX
				write(fd, &cmd, 1);
			}
		}
		if (incer) {
			incer = 0;
			grab_new_frame(w);

			delay = timer_ms() - last_move;
			if (delay >= alert_levels[alert_level+1].delay) {
				alert_level++;
				run_alert(alert_level);
			}
		}
	} while(n > 0);
	
	XDestroyWindow(ygl.display, w);
	YClose();
	capture_close();
	return(0);
}
