/*****************************************************************************
 *                                                                           *
 *            PROBAL := Probabilistic Alignment   v 0.1rc2                   *
 *            --------------------------------------------                   *
 *                                                                           *
 *            Needleman-Wunsh global affine-gap alignment                    *
 *                 over modified probabilistic space                         *
 *              of the Thorne-Kishino-Felsenstein model                      *
 *                                                                           *
 *                    AUTHOR :: Marek Kochanczyk                             *
 *                              Studies of Natural Sciences,                 *
 *                              Institute of Physics,                        *
 *                              Jagiellonian Unversity,                      *
 *                              Krakow, Poland                               *
 *                              kochanczyk(intron)corcoran.if.uj.edu.pl      *
 *                                                                           *
 *                      DATE :: 15 Aug 2004                                  *
 *                                                                           *
 *****************************************************************************/

#include<stdio.h>
#include<stdlib.h>
#include<time.h>
#include<png.h>
#include<math.h>

#define PPM_MAX 255
#define PPM_MAX_2 510
#define MATCH 1
#define MISMATCH -1
#define GAP_OPEN -2
#define GAP_EXTEND -1
#define NUM_OF_ALIGN 1

#define NEEDLEMAN 01
#define HIRSCHBERG 02
#define PPM 04
#define PNG 010
#define LOG_IT 020


typedef struct {
	float ** bg;
	float min;
	float max;
} s_tkfbg;


int max(int a, int b)
{
	if (a > b) {
		return a;
	} else {
		return b;
	}
}


void writepng(FILE *dfile, unsigned char * img, int xs, int ys)
{
	png_structp png_ptr;
	png_infop info_ptr;
	png_bytep *row_pointers;
	png_textp text_ptr;
	int y;
 
	/* Create and initialize the png_struct with the default error handlers */
	png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (png_ptr == NULL) {
		fprintf(stderr,"Failed to write PNG file");
		return; /* Could not initialize PNG library, return error */
	}
 
	/* Allocate/initialize the memory for image information. */
	info_ptr = png_create_info_struct(png_ptr);
	if (info_ptr == NULL) {
		png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
		fprintf(stderr, "Failed to write PNG file");
		return; /* Could not initialize PNG library, return error */
	}

	/* Set error handling for setjmp/longjmp method of libpng error handling */
	if (setjmp(png_jmpbuf(png_ptr))) {
		/* Free all of the memory associated with the png_ptr and info_ptr */
		png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
		/* If we get here, we had a problem writing the file */
		fprintf(stderr,"Failed to write PNG file");
		return; /* Could not open image, return error */
	}
 
	/* Set up the input control if standard C streams in use */
	png_init_io(png_ptr, dfile);

	png_set_IHDR(png_ptr, info_ptr, xs, ys,
					8, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
					PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
					
	png_set_gAMA(png_ptr, info_ptr, 1.0);

	text_ptr = (png_textp) png_malloc(png_ptr, (png_uint_32)sizeof(png_text) * 2);

	text_ptr[0].key = "Description";
	text_ptr[0].text = "Needleman-Wunsch alignment over TKF probabilistic map";
	text_ptr[0].compression = PNG_TEXT_COMPRESSION_NONE;
#ifdef PNG_iTXt_SUPPORTED
	text_ptr[0].lang = NULL;
#endif
 
	text_ptr[1].key = "Software";
	text_ptr[1].text = "Probal v0.1 by Marek Kochanczyk";
	text_ptr[1].compression = PNG_TEXT_COMPRESSION_NONE;
#ifdef PNG_iTXt_SUPPORTED
	text_ptr[1].lang = NULL;
#endif
	png_set_text(png_ptr, info_ptr, text_ptr, 1);

	row_pointers = (png_bytep *) png_malloc(png_ptr, ys*sizeof(png_bytep));
	for (y=0; y<ys; y++) {
		row_pointers[y] = &img[y * xs * 3];
	}
 
	png_set_rows(png_ptr, info_ptr, row_pointers);

	/* single call to write the whole PNG file into memory */
	 png_write_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, NULL);

	png_free(png_ptr, row_pointers);
	png_free(png_ptr, text_ptr);

	/* clean up after the write and free any memory allocated */
	png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
 
	return;
}


void writeppm(FILE *dfile, unsigned char * img, int xs, int ys)
{
   if (img != NULL) {
		int y;
 
		fprintf(dfile,"%s\n","P6");
		fprintf(dfile,"%d\n", xs);
		fprintf(dfile,"%d\n", ys);
		fprintf(dfile,"%d\n",PPM_MAX);
   
		for (y=0; y<ys; y++) {
			fwrite(&img[xs*3*y], xs*3, 1, dfile);
		}
	}
}


float trans(char letter1, char letter2, float s, float * FREQ)
{
	return ( 1 - exp(s)*FREQ[(int)letter2] + ((letter1 == letter2)?1:0)*exp(s) );
}


s_tkfbg tkf(		const char *          A,        /* first sequence           */
						const char *          B,        /* second sequence          */
            		const unsigned int *  lenA,     /* length of the 1st seq    */
		            const unsigned int *  lenB,     /* length of tje 2nd seq    */
						float *               FREQ,     /* distribution frequency   */
						float                 tau,	     /* time horizon             */
						short unsigned int    logit)	  /* time horizon             */
{
	s_tkfbg tkfbg;                                 /* struct with bg & xtremas */
	float ** V;                                    /* prefix-derived probabili */
	float ** W;                                    /* suffix-derived probabili */
	float VWmin=1000.0f, VWmax=-1000.0f;           /* extremas for scaling     */
	unsigned int i, j;                             /* counters                 */
	
	float lambda = 0.03718f;                       /* TKF  coefficients ------ */
	float mu     = 0.03744f;                       
	float s      = 0.91618f;                       
/*	float exps   = exp(s);     <-- we don't need it this time                  */
	float sigma  = s/tau;
	double Beta  = (1-exp((lambda-mu)*tau)) / (mu-lambda*exp((lambda-mu)*tau));
	float alpha  = lambda/mu;
	float E      = mu*Beta;
	float N      = (1 - exp(-mu*tau) - mu*Beta)*(1 - lambda*Beta);
	float H      = exp(-mu*tau)*(1 - lambda*Beta);

	if ((V = (float **) malloc ((*lenA+1)*sizeof(float *))) == NULL) {
			fprintf(stderr,"ProbAl!! Nie udalo sie zainicjalizowac mapy TKF\n");
			exit(-1);
	} else {
		for (i=0; i< *lenA+1; i++)	{
			if ( ( *(V+i) = (float *)malloc((*lenB+1)*sizeof(float))) == NULL ) {
				fprintf(stderr,"ProbAl!! Nie udalo sie zainicjalizowac mapy TKF\n");
				exit(-1);
			}
		}
	}
	if ((W = (float **) malloc ((*lenA+1)*sizeof(float *))) == NULL) {
			fprintf(stderr,"ProbAl!! Nie udalo sie zainicjalizowac mapy TKF\n");
			exit(-1);
	} else {
		for (i=0; i< *lenA+1; i++)	{
			if ( ( *(W+i) = (float *)malloc((*lenB+1)*sizeof(float))) == NULL ) {
				fprintf(stderr,"ProbAl!! Nie udalo sie zainicjalizowac mapy TKF\n");
				exit(-1);
			}
		}
	}
	V[0][0] = 1;
	W[*lenA][*lenB] = 1;
	
	for(i=1;i<=*lenA;i++) {
		V[i][0] = V[i-1][0];
		W[*lenA-i][*lenB] = W[*lenA-i+1][*lenB];
	}
	for(j=1;j<=*lenB;j++) {
		V[0][j] = V[0][j-1];
		W[*lenA][*lenB-j] = W[*lenA][*lenB-j+1];
	}
	
	for(i=0;i<*lenA;i++) {
		for(j=0;j<*lenB;j++) {
				  
/*	simplified version, without influence of freq(A_i)
 *	(requires transition probability function 'trans', independent on freq(A_i))*/
 	
			V [i+1][j+1] = V[i  ][j+1]*E                               \
				+ V[i+1][j  ]*E                                         \
				+	V[i  ][j  ]                                          \
				* ((N + H * trans(*(A+i),*(B+j), sigma*tau, FREQ)) - E*E);
			
			W [*lenA-1-i][*lenB-1-j] = W[*lenA-i  ][*lenB-j-1]*E       \
				+ W[*lenA-i-1][*lenB-j  ]*E                             \
				+ W[*lenA-i  ][*lenB-j  ]                               \
				* (N + H * trans(*(A+i),*(B+j), sigma*tau, FREQ) - E*E);
/* original TKF-model version
 * (doesn't require trans - dependent on FERQ matrix)

			V[i+1][j+1] = V[i  ][j+1] * alpha * FREQ[(int)*(A+i)] * E                                       \
		      + V[i  ][j  ] * alpha * FREQ[(int)*(A+i)]                                                    \
				* ( N  * FREQ[(int)*(B+j)] + H * ( (1-exps) * FREQ[(int)*(B+j)]*FREQ[(int)*(A+i)]            \
							+ exps*((*(A+i)==*(B+j))?FREQ[(int)*(A+i)]:0) ) )                                   \
							+ ( V[i+1][j  ]- V[i][j]*alpha*FREQ[(int)*(A+i)]*E ) * alpha * FREQ[(int)*(B+j)];
			
			W[*lenA-1-i][*lenB-1-j] = W[*lenA-i  ][*lenB-j-1] * alpha * FREQ[(int)*(A+*lenA-1-i)] * E        \
					      + W[*lenA-i  ][*lenB-j  ] * alpha * FREQ[(int)*(A+*lenA-1-i)]                        \
							* ( N  * FREQ[(int)*(B+*lenB-j-1)]                                                   \
						       + H * (  (1-exps)*FREQ[(int)*(A+*lenA-i-1)]*FREQ[(int)*(B+*lenB-j-1)]            \
					          + exps*((*(A+*lenA-i-1)==*(B+*lenB-j-1))?FREQ[(int)*(A+*lenA-i-1)]:0)            \
								 * FREQ[(int)*(A+*lenA-i-1)] ))                                                   \
							+ ( W[*lenA-i-1][*lenB-j  ]- W[*lenA-i][*lenB-j]*alpha*FREQ[(int)*(A+*lenA-i-1)]*E ) \
 							* alpha * FREQ[(int)*(B+*lenB-j-1)];
*/
		}
	}
	
	if (logit) {
		for(i=0;i<*lenA;i++) {
			for(j=0;j<*lenB;j++) {
				if (abs(V[i+1][j+1]) > 3.4e-38) {
					V[i+1][j+1] = -logf(V[i+1][j+1]); 
				} else {
					V[i+1][j+1] = 100;
				}
				if (abs(W[*lenA-i-1][*lenB-j-1]) > 3.4e-38) {
					W[*lenA-i-1][*lenB-j-1] = -logf(W[*lenA-i-1][*lenB-j-1]);
				} else {
					W[*lenA-i-1][*lenB-j-1] = 100;
				}
			}
		}
	}
	for(i=1;i<*lenA+1;i++) {
		for(j=1;j<*lenB+1;j++) {
			V[i][j] += W[i-1][j-1];
			if ( V[i][j] > VWmax){
				VWmax = V[i][j];
			}
			if ( V[i][j] < VWmin){
				VWmin = V[i][j];
			}
		}
	}

	tkfbg.bg = V;
	tkfbg.max = VWmax;
	tkfbg.min = VWmin;

	return tkfbg;
}


long int needleman(
	const char *         A,                       /* first sequence            */
   const char *         B,                       /* second sequence           */
	const char *         A_head,                  /* header of the 1st seq     */
	const char *         B_head,                  /* header of the 2nd seq     */
	const unsigned int * lenA,                    /* length of the 1st seq     */
	const unsigned int * lenB,                    /* length of the 2nd seq     */
	long int **          NW,                      /* main NWunsch array        */
	unsigned int ***     NW_trace,                /* NWunsch backtracing       */
	short int            MX[][128],               /* substitution matrix       */
	short int            gap_open,                /* penalty for opening a gap */
	short int            gap_extend,              /* penalty for xtending a gap*/
	unsigned int *       alignment_no)            /* current alignment id      */
{
	char nw_alignment[3][(*lenA) + (*lenB)];      /* needleman-wunsched seqen~s*/
	FILE * falign;                                /* file w. every last alignm */
	char falign_name[64];                         /* name of that file         */
	char stmp[64];                                /* temporary for string      */
	long int left=0, up=0, diagonal=0;            /* arrows in NW matrix       */
	long int tmp=0, i,j, t, dir;                  /* counter,temporaries       */
	short unsigned int gappresent = 0;            /* n(o), h(orizontal), v(~al)*/
	
	NW[0][0]=0;
	for(i=1; i <= *lenA; i++) {
		NW[i][0] = NW[i-1][0] + gap_extend;
	}
	for(j=1; j <= *lenB; j++) {
		NW[0][j] = NW[0][j-1] + gap_extend;
	}
	for(i=1; i<= *lenA; i++) {
		for(j=1; j <= *lenB; j++) {
				  
			left     = NW[i-1][j  ];			
			up       = NW[i  ][j-1];			
			diagonal = NW[i-1][j-1];

			dir = max(diagonal,max(left,up));
			
			if (dir == diagonal) {
				if (gappresent) {
					gappresent=0;
				}
				NW[i][j] = diagonal + MX[(int) *(A+i) ][(int) *(B+j) ];
			} else if (dir == left) {
				if (!gappresent) {
					gappresent=1;
					t = gap_open;
				} else {
					t = gap_extend;
				}
				NW[i][j] = left + t;
			} else if (dir == up) {
				if (!gappresent) {
					gappresent=1;
					t = gap_open;
				} else {
					t = gap_extend;
				}
				NW[i][j] = up + t;
			}
					
/* old non-affine gap penalty version if you prefer for any reasons:

	NW[i][j] = max ( 	NW[i-1][j  ] + gap_extend        ,
	                  max(	NW[i  ][j-1] + gap_extend  ,
	                        NW[i-1][j-1] + MX[(int)*(A+i)][(int)*(B+j)] ) );
*/		}
	}
	
	i = *lenA-1;
	j = *lenB-1;
	NW_trace[*alignment_no-1][0][0] = i;
	NW_trace[*alignment_no-1][0][1] = j;
	nw_alignment[0][0] = A[i];
	nw_alignment[1][0] = '!';
	nw_alignment[2][0] = B[j];
	
	t=0;
	while ((i>0) || (j>0)) {
		t++;
		if (i>0 && j>0) {
			left     = NW[i-1][j  ];
			up       = NW[i  ][j-1];
			diagonal = NW[i-1][j-1];
			tmp = max( left, max(up, diagonal) );
		} else {
			if (i==0) {
				tmp = up = NW[i  ][j-1];
				diagonal = left = up+1;
			} else if (j==0) {
				tmp = left = NW[i-1][j  ];
				diagonal = up = left+1;
			}
		}

		if (tmp == diagonal) {
				i--;
				j--;
				nw_alignment[0][t] = A[i];
				nw_alignment[1][t] = '|';
				nw_alignment[2][t] = B[j];
		} else if (tmp == left) {
				i--;
				nw_alignment[0][t] = A[i];
				nw_alignment[1][t] = ' ';
				nw_alignment[2][t] = '-';
		} else if (tmp == up) {
				j--;
				nw_alignment[0][t] = '-';
				nw_alignment[1][t] = ' ';
				nw_alignment[2][t] = B[j];
		} else {
				fprintf(stderr,"ProbAl:: needleman-wunsh zlapal wyjatek!\n");
		}
		NW_trace[*alignment_no-1][t][0] = i;
		NW_trace[*alignment_no-1][t][1] = j;
	}
	nw_alignment[1][t] = '!';
	
	strcpy(stmp, "needleman");
	sprintf(falign_name, "%s-%d.%ld", stmp, *alignment_no-1, NW[*lenA][*lenB]);
	if ( (falign = fopen(falign_name,"w")) == NULL) {
		perror("Nie zapisano pliku wynikowego z procedury uliniowenia metoda	Needlemana-Wunscha");
	} else {
		fprintf(falign,"Sekwencja 1: %s\n    Dlugosc: %d\n\n", A_head, *lenA);
		fprintf(falign,"Sekwencja 2: %s\n    Dlugosc: %d\n\n", B_head, *lenB);
		fprintf(falign,"Kara za    otwarcie przerwy = %d\n", gap_open);
		fprintf(falign,"Kara za kontynuacje przerwy = %d\n", gap_extend);
		fprintf(falign,"Wynik uliniowienia = %ld\n\n", NW[*lenA][*lenB]);
		
#define LINEW 50
		while (t/LINEW > 0){
			for(i=0;i<LINEW;i++) {
					putc(nw_alignment[0][t-i],falign);
			}
			putc('\n', falign);
			for(i=0;i<LINEW;i++) {
					putc(nw_alignment[1][t-i],falign);
			}
			putc('\n', falign);
			for(i=0;i<LINEW;i++) {
					putc(nw_alignment[2][t-i],falign);
			}
			putc('\n', falign);
			t-=LINEW;
			putc('\n', falign);
		}
			for(i=0;i<=t;i++) {
					putc(nw_alignment[0][t-i],falign);
			}
			putc('\n', falign);
			for(i=0;i<=t;i++) {
					putc(nw_alignment[1][t-i],falign);
			}
			putc('\n', falign);
			for(i=0;i<=t;i++) {
					putc(nw_alignment[2][t-i],falign);
			}
			putc('\n', falign);
			putc('\n', falign);
#undef LINEW
		fclose(falign);
		fprintf(stdout,"   [ok]  Uliniowanie zapisano do pliku \'%s\'\n",
			falign_name);
	}
	return ( NW[*lenA][*lenB] );
}


int main(int argc, char * argv[])
{
	unsigned short int FLAG = 00;                 /* bitwise option repository */
	clock_t start, end;                           /* time measuring            */
			  
	char * A;                                     /* first sequence            */
	char * B;                                     /* second sequence           */
	char * A_head;                                /* header of first sequence  */
	char * B_head;                                /* header of second sequen   */
	unsigned int lenA;                            /* length of first sequence  */
	unsigned int lenB;                            /* length of second sequence */
	
	long int ** alignment;                        /* alignment matrix          */
	unsigned int *** traceback;                   /* backtracing coords alignmx*/
	s_tkfbg  tkfbg;                               /* tkf-model image background*/
	short int MX[128][128];                       /* substitution matrix       */
	unsigned short int MX_size;						 /* substitution matrix size  */
	char MX_head[129];									 /* substitution matrix header*/
	float FREQ[128];                              /* bioletter frequency matrix*/
	short int gap_open=GAP_OPEN;                  /* penalty for  opening a gap*/
	short int gap_extend=GAP_EXTEND;              /* penalty for xtending a gap*/
	unsigned short int alignments=NUM_OF_ALIGN;   /* # of alignments to perform*/
	float argtau = -1.0;                          /* time horizon for tkf      */
	float tkfintegral = -1.0;                     /* @tkf normalized integral  */
	float ftmp = 0.0f;                            /* temporary float var       */
	
	int tmp;                                      /* counters, teporaries, ... */
	long int i,j;                                 /*                       ... */
	char letter;                                  /*                       ... */
	unsigned int size;                            /*                       ... */
	
	unsigned char * image;
	char * fsubmx_name = NULL;                    /* n: input : substitution mx*/
	char * ffreq_name  = NULL;                    /* n: input : freq matrix    */
	char   fout_name[32];                         /* n: output: image          */
	char   fplot_name[32];                        /* n: output: data for plot  */
	FILE * fA;                                    /* f: input : 1st sequence   */
	FILE * fB;                                    /* f: input : 2nd sequence   */
	FILE * fmx;                                   /* f: input : subst mx, freq */
	FILE * fout;                                  /* f: output: image          */
	FILE * fplot;                                 /* f: output: data for plot  */

/* Parsing command line call ------------------------------------------------ */

	if (argc == 1) {
		fprintf(stderr, "ProbAl, the probabilistic aligner, version 0.1, by Marek Kochanczyk\n\n");
		fprintf(stderr, "Usage:\n");
		fprintf(stderr, "  $> probal seq_1_file.fasta                                       \\\n");
		fprintf(stderr, "            seq_2_file.fasta                                       \\\n");
		fprintf(stderr, "            --matrix <PAM250 | BLOSUM62 | DNA | dna | other_file>  \\\n");
		fprintf(stderr, "            --tau <float#>                                         \\\n");
		fprintf(stderr, "            --freq <freq_file>                                     \\\n");
		fprintf(stderr, "          [ --aligns <int#> ]                                      \\\n");
		fprintf(stderr, "          [ --algo <nw | hir | auto> ]                             \\\n");
		fprintf(stderr, "          [ --gap-open <#> ]                                       \\\n");
		fprintf(stderr, "          [ --gap-extend <#> ]                                     \\\n");
		fprintf(stderr, "          [ --log ]                                                \\\n");
		fprintf(stderr, "          [ --pixmap <png | ppm> ]                                 \n\n");
		fprintf(stderr, "  $> probal --help                                                 \n\n");
		fprintf(stderr, "  $> probal --version\n\n"                                              );
		exit(0);
	}
	if (argc == 2) {
		if (!strcmp(argv[1], "--help")) {
			main(1, NULL);
			exit(0);
		} else if (!strcmp(argv[1], "--version")) {
			fprintf(stdout, "ProbAl, v0.1\n");
			exit(0);
		}
	} else {
		for (i=3; i<argc; i+=2) {
			if( !strcmp(argv[i],"--algo") ) {
				if( !strcmp(argv[i+1],"nw") ) {
					FLAG |= NEEDLEMAN;
				} else if ( !strcmp(argv[i+1],"hir") ) {
					FLAG |= HIRSCHBERG;
				} else if ( !strcmp(argv[i+1],"auto") ) {
					;
				} else {
					fprintf(stderr, "Probal:: Nieznany rodzaj algorytmu -- autokwalifikacja\n");
				}
			} else if( !strcmp(argv[i], "--gap-open") ) {
				gap_open = atoi(argv[i+1]);
			} else if( !strcmp(argv[i], "--gap-extend") ) {
				gap_extend = atoi(argv[i+1]);
			} else if( !strcmp(argv[i], "--aligns") ) {
				alignments = atoi(argv[i+1]);
			} else if( !strcmp(argv[i], "--matrix") ) {
				fsubmx_name = argv[i+1];
			} else if( !strcmp(argv[i], "--freq") ) {
				ffreq_name = argv[i+1];
			} else if( !strcmp(argv[i], "--tau") ) {
				argtau = atof(argv[i+1]);
			} else if( !strcmp(argv[i], "--log") ) {
				FLAG |= LOG_IT;
			} else if( !strcmp(argv[i], "--pixmap") ) {
				if ( !strcmp(argv[i+1],"png") ) {
					FLAG |= PNG;
				} else if ( !strcmp(argv[i+1],"ppm") ) {
					FLAG |= PPM;
				} else {
					FLAG |= PNG;
					fprintf(stderr, "ProbAl:: Nieznany rodzaj pliku graficznego -- przyjto png\n");
				}
			} else {
				fprintf(stderr, "ProbAl:: Podano nieznany argument!\n");
				exit(-1);
			}
		} /* for */
		if (gap_open > 0) {
			fprintf(stderr, "ProbAl!! (--gap-open ?) Kara za otwarcie przerwy nie moze byc dodatnia\n");
			exit(-1);
		}
		if (gap_extend > 0) {
			fprintf(stderr, "ProbAl!! (--gap-extend ?) Kara za przedluzenie przerwy nie moze byc dodatnia\n");
			exit(-1);
		}
		if (argtau < 0) {
			fprintf(stderr, "ProbAl!! (--tau ?) Nalezy podac dodatni horyzont czasowy\n");
			exit(-1);
		}
		if ( fsubmx_name == NULL ) {
			fprintf(stderr, "ProbAl!! (--matrix ?) Nie wyspecyfikowano macierzy substytucji\n");
			exit(-1);
		}
		if ( ffreq_name == NULL ) {
			fprintf(stderr, "ProbAl!! (--freq ?)Nie wyspecyfikowano pliku czestotliwosci\n");
			exit(-1);
		}
	} /* if */
	
/* Reading file with 1st sequence ------------------------------------------- */
 
	fA = fopen(argv[1], "r");

	if ( (letter = getc(fA)) != '>' ) {
		fprintf(stderr, "ProbAl!! Pierwsza sekwencja musi by w formacie FASTA\n");
		exit(-1);
	}
	
	for (size = 2; (letter = getc(fA)) != EOF; size++) {
		if ( letter == '\n' ) {
			break;
		}
	}
	if ( ( A_head = (char *) malloc(size * sizeof(char)) ) == NULL ) {
		perror("ProbAl!! Inicjalizacja nagwka pierwszej sekwencji\n");
		exit(-1);
	}
	
	for (size = 0; (letter = getc(fA)) != EOF; ) {
		if ( letter != '\n' ) {
			size++;
		}
	}
	
	if ( ( A = (char *) malloc((size+1) * sizeof(char)) ) == NULL ) {
		perror("ProbAl!! Inicjalizacja pierwszej sekwencji\n");
		exit(-1);
	}

	rewind(fA);

	letter = getc(fA);
	for (i = 0; (letter = getc(fA)) != EOF && letter != '\n'; ) {
		A_head[i++] = letter;
	}
	A_head[i] = '\0';
	
	for (i = 0; (letter = getc(fA)) != EOF;) {
		if ( letter != '\n' ) {
			A[i++] = letter;
		}
	}
	A[i] = '\0';
	
	(void) fclose(fA);
	lenA = i;
	
/* Reading file with 2nd sequence ------------------------------------------- */

	fB = fopen(argv[2], "r");

	if ( (letter = getc(fB)) != '>' ) {
		fprintf(stderr, "ProbAl!! Druga sekwencja musi by w formacie FASTA");
		exit(-1);
	}
	
	for (size = 2; (letter = getc(fB)) != EOF; size++) {
		if ( letter == '\n' ) {
			break;
		}
	}
	if ( ( B_head = (char *) malloc(size * sizeof(char)) ) == NULL ) {
		perror("ProbAl!! Inicjalizacja nagwka drugiej sekwencji");
		exit(-1);
	}
	
	for (size = 0; (letter = getc(fB)) != EOF; ) {
		if (letter != '\n') {
			size++;
		}
	}
	if ( ( B = (char *) malloc((size+1) * sizeof(char)) ) == NULL ) {
		perror("ProbAl!! Inicjalizacja drugiej sekwencji");
		exit(-1);
	}

	rewind(fB);
	
	letter = getc(fB);
	for (i = 0; (letter = getc(fB)) != EOF && letter != '\n'; ) {
		B_head[i++] = letter;
	}
	B_head[i] = '\0';
	
	for (i = 0; (letter = getc(fB)) != EOF; ) {
		if (letter != '\n') {
			B[i++] = letter;
		}
	}

	(void) fclose(fB);
	lenB = i;
	
/* Initializing substitution matrix ----------------------------------------- */

	for (i=0; i<128; i++) {
		for (j=0; j<128; j++) {
			if (i == j) {
				MX[i][j] = MATCH;
			} else {
				MX[i][j] = MISMATCH;
			}
		}
	}
	MX['A']['a'] = MX['a']['A'] = MATCH;
	MX['T']['t'] = MX['t']['T'] = MATCH;
	MX['G']['g'] = MX['g']['G'] = MATCH;
	MX['C']['c'] = MX['c']['C'] = MATCH;
	MX['N']['n'] = MX['n']['N'] = MATCH;

	if ( (fmx = fopen(fsubmx_name, "r")) == NULL) {
		perror("ProbAl!! Nie wczytano macierzy substytucji");
		fprintf(stderr,"Uzywam domyslnej, gdzie homologia = %d, jej brak = %d\n",
			MATCH, MISMATCH);
	} else {
		fscanf(fmx, "%s", MX_head);
		MX_size = strlen(MX_head);
				  
		for (i=0; i<MX_size; i++) {
			for (j=0;j<=i;j++) {
				fscanf(fmx, "%d", &tmp);
				MX[ (int)MX_head[j] ][ (int)MX_head[i] ] = MX[ (int)MX_head[i] ][ (int)MX_head[j] ] = tmp;
			}
		}
		fclose(fmx);
	}

/* Initializing frequency table --------------------------------------------- */

	if ( (fmx = fopen(ffreq_name, "r")) == NULL) {
		perror("ProbAl!! Nie wczytano pliku czestotliwosci wystepowania bio-liter");
		exit(-1);
	} else {
		for (i=0; i<128; i++) {
			FREQ[i] = 0.0f;
		}
		while (!feof(fmx)) {
			fscanf(fmx, "%c", &letter);
			fscanf(fmx, "%f", &FREQ[(int) letter]);
		}
		fclose(fmx);
	}
	
/* Creating probability map due to modified TKF ----------------------------- */
	
	if ((image = (unsigned char *) malloc ((3*lenA*lenB)*sizeof(unsigned char ))) == NULL){
		perror("ProbAl!! alokowano pamiec dla pliku graficznego");
		exit(-1);
	}
	
	fprintf(stdout,"ProbAl:: Przygotowuje tlo prawdopodobienstwa wg modelu TKF... ");
	fflush(stdout);
	start = clock();
	tkfbg = tkf(A,B,&lenA,&lenB,FREQ,argtau,FLAG & LOG_IT);
	end = clock();
	fprintf(stdout,"[ok]\n         (przygotowanie tla zajelo %.2f sekundy)\n",
		((double)(end - start)) / CLOCKS_PER_SEC );
	
	fprintf(stdout,"ProbAl:: Skalowanie wartosci mapy tkf...");
	fflush(stdout);
	size=0;

	for(j=1;j<lenB+1;j++) {
		for(i=1;i<lenA+1;i++) {
			letter = ((FLAG & LOG_IT)?-1:1) * PPM_MAX*( tkfbg.bg[i][j] - tkfbg.min ) / (tkfbg.max - tkfbg.min);
			*(image+size++) = letter;
			*(image+size++) = letter;
			*(image+size++) = letter;
		}
	}
	fprintf(stdout," [ok]\n");
	
/* Performing classical alignment ------------------------------------------- */
	
	if ( !(FLAG & NEEDLEMAN) && !(FLAG & HIRSCHBERG)) {
		FLAG |= NEEDLEMAN;
		if ((alignment = (long int **) malloc ((lenA+1)*sizeof(long int *))) == NULL){
			FLAG &= ~NEEDLEMAN;
			FLAG |= HIRSCHBERG;
			fprintf(stdout,"ProbAl:: Automatycznie wybrano algorytm Needlemana-Wunscha");
		} else {
			for (i=0; i< lenA+1; i++)	{
				if ( ( *(alignment+i) = (long int*)malloc((lenB+1)*sizeof(long int))) == NULL ) {
					FLAG &= ~NEEDLEMAN;
					FLAG |= HIRSCHBERG;
					fprintf(stdout,"ProbAl:: Automatycznie wybrano algorytm Hirschberga (za malo pamieci dla N-W)\n");
					break;
				}
			}
		}
	}
	if ( (  traceback = (unsigned int ***) malloc(alignments*sizeof(unsigned int **))  ) == NULL ) {
		perror("\rProbAl!! Nie ustanowilem tablicy sledzenia wtornego\n");
		exit(-1);
	}
	for (i=0; i<alignments; i++) {
		if ( (  traceback[i] = (unsigned int **) malloc( ((lenA+lenB+1)*sizeof(unsigned int *)) )  ) == NULL ) {
			perror("\rProbAl!! Nie ustanowilem tablicy sledzenia wtornego\n");
			exit(-1);
		}
	}
	for (i=0; i<alignments; i++) {
		for (j=0; j<lenA+lenB+1; j++) {
			if ( ( traceback[i][j] = (unsigned int *) calloc(1, 2*sizeof(unsigned int)) ) == NULL )  {
				perror("\rProbAl!! Nie ustanowilem tablicy sledzenia wtornego\n");
				exit(-1);
			}
		}
	}
	
	if (FLAG & NEEDLEMAN) {
		fprintf(stdout, "ProbAl:: Uliniowanie algorytmem Needlemana-Wunscha... :\n");
		size = 1;
		tmp=1;
		start = clock();
		do {
			needleman( A, B, A_head, B_head, &lenA, &lenB, alignment, traceback, MX, gap_open, gap_extend, &size );
			if (!(tmp++ % 2)) {
				gap_open--;
			} else {
				gap_extend--;
			}
		} while (size++ < alignments);
		end = clock();
		fprintf(stdout,"         (uliniowani%c zajel%c %.2f sekundy)\n", \
				  (alignments>1)?'a':'e', ((alignments>1)?'y':'o'), ((double)(end - start)) / CLOCKS_PER_SEC );
				
	} else if (FLAG & HIRSCHBERG) {
		fprintf(stdout, "ProbAl:: Uliniowanie algorytmem Hirschberga...\n");
		fprintf(stdout, "ProbAl!! ALGORYTM WCIAZ NIE ZAIMPLEMENTOWANY!!!\n");
		exit(0);
	}

/* Storing gathered data in files ------------------------------------------- */
	
	fprintf(stdout, "ProbAl:: Zapisywanie plikow z danymi wykresow... :\n");
	for(i=alignments-1; i>= 0;i--) {

		strcpy(fplot_name, "plot");
		sprintf(fplot_name, "%s-%d", fplot_name,alignments-1-(int)i);
		tkfintegral = ftmp = 0.0f;
		if ((fplot = fopen(fplot_name, "w")) == NULL) {
			fprintf(stderr,"ProbAl!! Nie zapisano pliku \'%s\'\n", fplot_name);
		} else {
			for(j=0; traceback[i][j][0]!=0 || traceback[i][j][1]!=0; j++) {
				fprintf( fplot, "%ld \t%f\n", j,
					ftmp=((FLAG & LOG_IT)?-1:+1)*(tkfbg.bg[traceback[i][j][0]+1][traceback[i][j][1]+1]-tkfbg.min)/(tkfbg.max-tkfbg.min));
				tkfintegral += ftmp;				
			}
			fprintf(fplot, "# tkfintegral = %f\n", tkfintegral/j);
			fclose(fplot);
			fprintf(stdout, "   [ok]  Zapisano dane dla wykresu w \'%s\'\n", fplot_name);
		}
		for(j=0; j< lenA+lenB+1;j++) {
			image[3*(lenA*traceback[i][j][1]+traceback[i][j][0])  ] = PPM_MAX-i*(PPM_MAX/alignments);
			image[3*(lenA*traceback[i][j][1]+traceback[i][j][0])+1] = 0;
			image[3*(lenA*traceback[i][j][1]+traceback[i][j][0])+2] = PPM_MAX*i/alignments;
		} 
	}
	fprintf(stdout, "ProbAl:: Zapisywanie plikow z obrazem mapy i uliniowienia... ");
	fflush(stdout);
	
	strcpy(fout_name,"nw-tkf");
	if ( FLAG & PPM ) {
		sprintf(fout_name, "%s.ppm", fout_name);
	} else if ( FLAG & PNG ) {
		sprintf(fout_name, "%s.png", fout_name);
	} else {
		sprintf(fout_name, "%s.out", fout_name);       /* tho' will not be used */
	}
	
	if ( (fout = fopen(fout_name, "w")) == NULL) {
		fprintf(stderr, "Nie moglem zapisac pliku z obrazem\n");
	} else {
		if ( FLAG & PPM ) {
			writeppm(fout, image, lenA, lenB);
		} else if ( FLAG & PNG ) {
			writepng(fout, image, lenA, lenB);
		} else {
			fprintf(stderr," [brak arg --pixmap => NIE]\n");
		}
		fclose(fout);
		fprintf(stdout, "[ok]\n");
	}
	
	return 0;
}

