program main;
const
  MaxMltDataPoints      = 1001;
  MaxMapDataPoints      = 60060;
  MaxProDataPoints      = 60;

type
  MltDataPointType      = Record
                            Temp     : Real;
                            DthetaDT : Double;
                          end; { Record }
  MltDataPointArrayType = Array[1..MaxMltDataPoints] of MltDataPointType;

  MapDataPointType      = Record
                            Temp,
                            Pos   : Real;
                            Theta : Double;
                          end; { Record }
  MapDataPointArrayType = Array[1..MaxMapDataPoints] of MapDataPointType;

  ProDataPointType      = Record
                            Temp,
                            Pos   : Real;
                          end; { Record }
  ProDataPointArrayType = Array[1..MaxProDataPoints] of ProDataPointType;

var
   index, DomainsPerBlock : Longint;
   DOMAIN,
   MltDataPointNumber,
   LastMltDataPoint,
   MapDataPointNumber,
   ProDataPointNumber    : LongInt;
   TotalBasesMelted,
   MltDataSum,
   PARAMC                :Real; {INPUT}
   BlockMltActive        : Boolean;
   SALT,
   TSTEP,
   SIGMAP,
   TS,
   TF                    : Double;
   OutFile1Name,
   OutFile2Name,
   OutFile3Name,
   LogFileName,
   TempString,
   strindex,
   outputdir,
   basestr,
   TotalBasesMeltedString:String;
   TotalMltDataPoint,                             { Array for total melting curve }
   BlockMltDataPoint     : MltDataPointArrayType; { Array for block melting curve }
   MapDataPoint          : MapDataPointArrayType; { Array for positional plot }
   ProDataPoint          : ProDataPointArrayType; { Array for profile }
   InFile,
   OutFile1,
   OutFile2,
   OutFile3,
   LogFile               : text;
   Pchar                 : char;
   Pstring               : array[1..8] of char;
   Pval                  : Real;
   ParamFile : text;
   Names, InName, OutBase : string;
   
   

(******************************************************************************)
procedure MeltAfile(Const InFileName : String; InFileSize: Cardinal);

const
  MaxDNAblock           = 10000;

var
  X                     : Longint;
  NumOfMltDataPoints,
  NumOfMapDataPoints : LongInt;
  PrevBasesRead,
  TotalBasesRead        : Real;
  AutoScale,
  AutoPrint             : Boolean;


  (**************************************************************************)


  procedure MeltAblock; { looped melting program }


{
Translated from FORTRAN77 to Delphi/Pascal by J.W. Bizzaro, 1996.

Program MeltSim: J.Knowles & R.D.Blake, 1980.
This program is designed to simulate dna melting
using Poland's algorithm [Poland,D. (1974)
Biopolymers, 3, 1859], with the approximation
of loop entropy described By Fixman,M & Friere,J.
'Theory Of Dna Melting Curves'
Biopolymers, 16, 2693-2704. 977

Some definitions of variables and parameters:
TI: initial temperature of simulated curve, deg c
TF: final temperature of simulated curve
TSTEP: temperature increment for the simulated curve
NITS: total number of output thetas over the full
   temp range
DOMAIN: resolution of 3d denaturation map in base pairs
RES3D: resolution of theta set outputs for 3d plots
   (TEMP)
RESD0: resolution of d(THETA) in temp units
SIGMA: weight for cooperativity in helix interuption
SIGSTP: resolution in sigma, degrees
DSTART: weight for ssDNA stiffness at TI
DFINAL: weight for ssDNA stiffness at TF
DSTEP: resolution in D, degrees
ALPHA: loop closure exponent
END: weight for excess electrostatienergy at h-c
    boundary
NMAX: length of the sequence
IT: initial temperature for computation of first
    probability
TEMP: real value of the temperature
R: stability factors; weights for different NM
A: array of first coefficients for F/F exponential
LEF: loop energy function
B: array of second coefficients for f/f exponential
NEXP: number of exponentials used to simulate loop
    function

The program produces data for plotting the differential
melting curve and the denaturation map.

 Revisions :  August 1984, By G.Day
              June 1986, By J.D.Blake
              November 1988, By R.D.B.
              July 1991, R.D.B
              March 1993, R.D.B.
}

  label IncreaseIT;

  type
    DNAarrayType               = Array[0..MaxDNAblock] of Longint;
    RealArrayType6             = Array[0..6] of Real;
    DoubleArrayType12          = Array[0..12] of Double;
    DoubleArrayType16          = Array[0..16] of Double;
    ParrayType                 = Array[0..MaxDNAblock] of Double;

  var
    I,
    IESS,
    IT,
    N,
    N1,
    N2,
    NEXPS,
    M,
    RESD0,
    RES3D,
    X                          : Longint;
    BlockLENGTH,
    NMAX                       : LongInt;
    BETA,
    D0,
    DFINAL,
    DSTART,
    DSTEP,
    DTT,
    GAMMA,
    Log10SALT,
    NITS,
    P1,
    PM,
    Q,
    SIGMAI,
    SIGMAF,
    SIGSTP,
    SIGMA,
    SUM,
    TEMP,
    TFINAL,
    THETA,
    THETA0,
    TI,
    TM,
    TM0,
    TM1,
    TSTART,
    TPRIN,
    W                          : Double;
    BlockLENGTHstring          : String;
    SUMW                       : RealArrayType6;
    A,
    A0,
    B,
    D,
    E                          : DoubleArrayType12;
    R,
    TEMPS                      : DoubleArrayType16;
    DNA                        : DNAarrayType;
    P                          : ParrayType;
    RECALC                     : Boolean;


    (**********************************************************************)


    Function F(NatJ : Double) : Double;

{
Translated from FORTRAN77 to Delphi/Pascal by J.W. Bizzaro, 1996.

Evaluation of parameters in loop closure function
the following represents the loop entropy function
}

    begin { F }
       {PARAMC:=1.75;}
      F := Exp(Ln(NatJ + 1.0) * (-PARAMC)); { (N(J)+1)^-1.75 }

    end; { F }


    (*********************************************************************)


    Procedure LEF;

{
Translated from FORTRAN77 to Delphi/Pascal by J.W. Bizzaro, 1996.

Routine to compute the parameters for the loop closure function.

Written by Gary R. Day.
Latest revision: June 5,1983

This program is designed to return the parameter values of
A0(J),B(J) J=1,2,...,NEXPS such that the summation of
A0(J)*EXP(-B(J)*N) J=1,2,...,NEXPS  approximates the
function: D(N) = F(N)

The program places as many exponentials as possible under the curve
for the given values of SIGMA, D0, ALPHA, AND NMAX where NMAX is
maximum N.  Up to 12 exponentials will be placed under the curve
Each exponential is fitted through through two points on the curve
These points are selected using the geometric series:
N(J) = ((NMAX+D0)*R**J)-D0 J=0,1,...,2NEXPS-1  WHERE R IS CHOSEN SO THAT
(NMAX+D0)*R**(2NEXPS-1) = D0+1.

This algorithm is described in:...
M. Fixman And J. Ree  'Theory Of Dna Melting Curves'
"Journal Of Biopolymers" Volume 16 Pp. 2693-2704. 977

Input parameters...
     SIGMA,NMAX,D0,ALPHA
output parameters...
    NEXPS = number of exponentials
    A0 = an array dimensioned at least 12
    B = an array dimensioned at least 12
}

    label
      StartAgain,
      AnotherIteration;

    type
      DoubleArrayType24            = Array[0..24] of Double;

    var
      NEXPS2,
      J,
      K,
      L,
      M                          : Longint;
      DIFF1,
      DIFF2,
      DET,
      R,
      SUM1,
      SUM2                       : Double;
      OLDA0,
      OLDB                       : DoubleArrayType12;
      DELTA,
      N                          : DoubleArrayType24;

    begin { LEF }
    { Set the parameters to zero. }
      NEXPS := 13;
{ 20 }
    StartAgain:
      NEXPS := NEXPS - 1;

      for J := 1 to NEXPS do
      begin
        A0[J] := 0;
{ 30 }
        B[J] := 0;
      end;

{
Pick the data points from a geometric series NMAX*R**J such that
(NMAX+D0)*R**(2*NEXPS-1) = D0..
}

      NEXPS2 := NEXPS * 2;

      R := Exp(Ln((D0 + 1.0) / (NMAX + D0)) * (1.0 / (NEXPS2 - 1)));

      for J := 1 to NEXPS2 do
      begin

        N[J] := (NMAX + D0) * Exp(Ln(R) * (NEXPS2 - J)) - D0;
{ 40 }
        DELTA[J] := F(N[J]);
      end;

{ Save the current values for future reference. }

{ 50 }
    AnotherIteration:
      for J := 1 to NEXPS do
      begin
        OLDA0[J] := A0[J];
{ 60 }
        OLDB[J] := B[J];
      end;

{
Fit the exponentials one at a time taking 2 points at a time from
2 NEXPS points starting with points (2NEXPS,2NEXPS-1).
}

      L := NEXPS2 + 1;

      for M := 1 to NEXPS do
      begin
        J := NEXPS - M + 1;
        L := L - 2;

{ First calculate the sums of the other exponentials at the two PO }

        SUM1 := 0;
        SUM2 := 0;

        for K := 1 to NEXPS do
        begin
          if (K <> J) and (A0[K] <> 0) { check for exponential underflow. }
            then
            begin
              if ((B[K] * N[L]) <= 180.0)
               and ((-Ln(A0[K]) + B[K] * N[L]) <= 180.0)
                then
                  SUM1 := SUM1 + A0[K] * Exp(-B[K] * N[L]);
              if ((B[K] * N[L + 1]) <= 180.0)
               and ((-Ln(A0[K]) + B[K] * N[L + 1]) <= 180.0)
                then
                  SUM2 := SUM2 + A0[K] * Exp(-B[K] * N[L + 1]);
            end;
{ 75 }
        end;

{
Calculate the differences between DELTA(N(L) AND DELTA(N(L+1)) a
corresponding sums.
}

        DIFF1 := DELTA[L] - SUM1;
        DIFF2 := DELTA[L + 1] - SUM2;

{ If there are too many exponentials, drop one and start again }

        if (DIFF1 <= 0) or (DIFF2 <= 0)
          then
            Goto StartAgain; { 20 }

{ Solve for B(J) and the log of A0(J). }

        DIFF1 := Ln(DIFF1);
        DIFF2 := Ln(DIFF2);
        DET := N[L] - N[L + 1];
        A0[J] := Exp((-DIFF1 * N[L + 1] + DIFF2 * N[L]) / DET);
        B[J] := (DIFF2 - DIFF1) / DET;
{ 200 }
      end; { for }


{ Test for convergence and perform another iteration if neccessary }

      for J := 1 to NEXPS do
      begin
        if (Abs(A0[J] - OLDA0[J]) / A0[J] > 1.0E-08)  or
         (Abs(B[J] - OLDB[J]) / B[J] > 1.0E-08)
          then
            Goto AnotherIteration; { 50 }
{ 400 }
      end;
    end; { LEF }


    (**********************************************************************)


    Procedure GETDNA;

{
Translated from FORTRAN77 to Delphi/Pascal with major changes
by J.W. Bizzaro, 1996.

Read in the file header and the DNA sequence and determine its length
Give DNA(M) an integral value between 1 and 16 indicating which
duplet DNA(M)DNA(M+1) represents.
}

    var
      Base          : Char;
      StartAtBase,
      EndAtBase     : Real;
      MeltByMarkedBlocks : Boolean;
      BlockSize    : Longint;

    begin { GETDNA }
      PrevBasesRead := TotalBasesRead;
      Base := ' ';
      BlockLENGTH := 0;
      StartAtBase := 1; {INPUT}
      EndAtBase := -1;   {INPUT, -1 for end}
      MeltByMarkedBlocks := false;
      BlockSize := 500;

{ 50 }
      While (not EOF(InFile))
        and (not (MeltByMarkedBlocks and (Base = '*')))
        and (BlockLENGTH < BlockSize) do
      begin
{ 75 }
        Read(Infile, Base);

        if Base in ['X', 'A', 'a', 'C', 'c', 'G', 'g', 'T', 't']
          then
          begin
            TotalBasesRead := TotalBasesRead + 1;
            if (TotalBasesRead >= StartAtBase)
             and ((TotalBasesRead <= EndAtBase)
             or (EndAtBase = -1))
              then
              begin
                BlockLENGTH := BlockLENGTH + 1;
                case Base of
                  'X' : DNA[BlockLENGTH] := 1;
                  'A' : DNA[BlockLENGTH] := 1;
                  'a' : DNA[BlockLENGTH] := 1;
                  'C' : DNA[BlockLENGTH] := 2;
                  'c' : DNA[BlockLENGTH] := 2;
                  'G' : DNA[BlockLENGTH] := 3;
                  'g' : DNA[BlockLENGTH] := 3;
                  'T' : DNA[BlockLENGTH] := 4;
                  't' : DNA[BlockLENGTH] := 4;
                end; { case }

                if (BlockLENGTH > 1)
                  then
                    DNA[BlockLENGTH - 1] := (DNA[BlockLENGTH - 1] - 1) * 4 + DNA[BlockLENGTH];
              end; { if }
          end; { if }
{ 100 }
{ 200 }
{ 250 }
      end; { While }
    end; { GETDNA }


    (**********************************************************************)


  begin { MeltAblock }
    { (Re-)initialize MapDataPoint array...last block only }
    for X := 1 to MaxMapDataPoints do
    begin
      MapDataPoint[X].Temp := 0;
      MapDataPoint[X].Pos := 0;
      MapDataPoint[X].Theta := 0;
    end;

    { Initialize ProDataPoint array }
    for X := 1 to MaxProDataPoints do
    begin
      ProDataPoint[X].Temp := 0;
      ProDataPoint[X].Pos := 0;
    end;

    MltDataPointNumber := 1;
    MapDataPointNumber := 1;
    ProDataPointNumber := 1;

    Log10SALT := Ln(SALT) / Ln(10);

{ The following stabilities were determined in July 95 }

    { GG } TEMPS[11] := 13.90 * Log10SALT + 388.41;
    { CC } TEMPS[6] := 13.90 * Log10SALT + 388.41;
    { GC } TEMPS[10] := 13.80 * Log10SALT + 424.55;
    { CG } TEMPS[7] := 12.97 * Log10SALT + 376.12;
    { AC } TEMPS[2] := 15.50 * Log10SALT + 396.31;
    { GT } TEMPS[12] := 15.50 * Log10SALT + 396.31;
    { CA } TEMPS[5] := 17.22 * Log10SALT + 360.72;
    { TG } TEMPS[15] := 17.22 * Log10SALT + 360.72;
    { AG } TEMPS[3] := 14.59 * Log10SALT + 369.78;
    { CT } TEMPS[8] := 14.59 * Log10SALT + 369.78;
    { GA } TEMPS[9] := 20.24 * Log10SALT + 380.79;
    { TC } TEMPS[14] := 20.24 * Log10SALT + 380.79;
    { AT } TEMPS[4] := 16.95 * Log10SALT + 360.84;
    { TA } TEMPS[13] := 23.60 * Log10SALT + 353.27;
    { AA } TEMPS[1] := 20.02 * Log10SALT + 362.26;
    { TT } TEMPS[16] := 20.02 * Log10SALT + 362.26;

{
The variables <DOMAIN> and <RES3D> are used to establish the size
of the domain over which probabilities will be averaged for
the denaturation map. <RES3D> is an longint and refers
to the number of temperature increments between each set
of denaturation probabilities to be output. Thus if RES3D
is equal to 5 and TSTEP = .1, a set of values will be output
at every 1/2 degree, i.e. every 5th increment.
}
    { see below for DOMAIN }
    RES3D := 1;

{ Establish the temperature scan and increment for the simulation. }

    TM0 := 86.5677 + 20.1534 * Log10SALT;
    TM1 := 121.03 + 13.7075 * Log10SALT;

{ Establish the temperature scan and increment for the simulation. }

    TSTART := TS + 273.16;
    TFINAL := TF + 273.16;
    NITS := (TFINAL - TSTART) / TSTEP + 1.0 + TSTEP;

{
SIGMAI and SIGMAF are cooperativity parameters (equilibrium constants)
for start and final temperatures of the range TSTART to TFINAL
}
    SIGMAI := -5.22E-06 * (TS - TM0) / (TM1 - TM0) + 9.0E-06;
    SIGMAF := -5.22E-06 * (TF - TM0) / (TM1 - TM0) + 9.0E-06;

{
ALPHA is the loop closure exponent F(N)=(N+D)^(-ALPHA)
where D is stiffness, and N=number of base pairs
function= probability that the melting loop remains closed
as the number of base pairs melted increases, the closure is more in doubt
}

{ DSTART is D,the stiffness parameter at TSTART }

    DSTART := 1;
    DFINAL := 1;
    RESD0 := 1;

{
* IMPORTANT * RESD0 refers to the resolution of D0 in temperature
increments. For example if RESD0=5 and TSTEP=.1 then D0 will
be recalculated every 0.5 degrees, i.e. every 5th iteration.
If D0 is to be constant simply set DFINAL = DSTART and RESD0=1
}

    DSTEP := (DFINAL - DSTART) / (NITS / RESD0);
    SIGSTP := (SIGMAF - SIGMAI) / (NITS - 1);

{ Read in the (next) sequence }

    GETDNA;

    TI := TSTART - 273.16;
    TF := TFINAL - 273.16;
{ 52 }
{ 65 }
{ 66 }
    NMAX := (BlockLENGTH - 1) - 1;
    DOMAIN := Trunc(BlockLENGTH / DomainsPerBlock);
    TotalBasesMelted := TotalBasesMelted + BlockLENGTH;
    N1 := NMAX - 1;
    N2 := NMAX - 2;
    P[NMAX] := 0;
    THETA0 := 1.0;
    SUM := 0;
    IESS := 0;
    SIGMA := SIGMAI;
    D0 := DSTART;
    RECALC := TRUE;
    IT := -11;
{ 1000 }

  { Raise the temperature and increase the mltdatapoint number }
  IncreaseIT:
    IT := IT + 1;
    TEMP := TSTART + IT * TSTEP;
{
Assign values to the parameters of the loop entropy factor and
calculate the A'S and B'S.
}

    if RECALC
      then
      begin
        if (IT > 0)
          then
            D0 := DSTART + IT / RESD0 * DSTEP;

        LEF;

        for I := 1 to NEXPS do
{ 1050 }
          B[I] := Exp(-B[I]);

        RECALC := FALSE;
      end;

{ 1075 }
    if (DSTEP <> 0) and (IT > 0) and (((IT + 1) MOD RESD0) = 0)
      then
        RECALC := TRUE;

    if (IT > 0)
      then
        SIGMA := SIGMAI + IT * SIGSTP;

    for I := 1 to NEXPS do
{ 1080 }
      A[I] := A0[I] * SIGMAP;

{
Calculate R(M) - the helix to coil stability factor at M.
DNA(I) is the index of the stability of base pair in degrees
Kelvin.
}

    for I := 1 to 16 do
{ 1100 }
      R[I] := Exp(12.5047 * (1.0 - (TEMPS[I] / TEMP)));

{
Based on an entropy per basepair of 24.85 cal/mol-bp.deg.
Apply Poland's reverse recursion formula to calculate the
conditional probability that the (M+1)th base pair is in the
helix state given that the Mth pair is in the helix state.
}

    for I := 1 to NEXPS do
{ 1150 }
      E[I] := 0;

    BETA := R[DNA[NMAX]];
    TM := R[DNA[N1]] / (1.0 + R[DNA[NMAX]]);
    P[N1] := 1.0 / (1.0 + BETA);

    for N := 1 to N2 do
    begin
      M := N2 - N + 1;
      Q := 0;

      for I := 1 to NEXPS do
      begin
        E[I] := TM * B[I] * (1.0 + E[I]);
{ 1200 }
        Q := Q + A[I] * E[I];
      end;

      BETA := TM * BETA;
      P[M] := 1.0 / (1.0 + Q + BETA);
{ 2000 }
      TM := R[DNA[M]] * P[M];
    end;

{
Now to calculate the unconditional probabilities we apply
Poland's forward recursion formula.
}

    THETA := 0;
    GAMMA := R[DNA[1]] * P[1];
    P1 := 1.0 + GAMMA;

    for M := 2 to N1 do { Go thru all bases }
    begin
      GAMMA := R[DNA[M]] * P[M] * GAMMA;
{ 2200 }
      P1 := P1 + GAMMA;
    end;

    P1 := 1.0 / P1;

    for I := 1 to NEXPS do
{ 2300 }
      D[I] := 0;

    GAMMA := R[DNA[1]] * P[1];
    PM := P1;

    for M := 1 to N1 do { Go thru all bases }
    begin
      W := 0;

      for I := 1 to NEXPS do
      begin
        W := W + A[I] * D[I];
{ 2600 }
        D[I] := R[DNA[M + 1]] * P[M + 1] * B[I] * (PM * P[M] + D[I]);
      end;

      PM := P1 * GAMMA + PM * P[M] + W;
      GAMMA := GAMMA * R[DNA[M + 1]] * P[M + 1];

{ The following section produces a melting map data point }

      if (IT >= 0) and (IT MOD RES3D = 0)
        then
        begin
          SUM := SUM + PM;

          { When M is a multiple of the DOMAIN, record a data point }
          if (M MOD DOMAIN = 0) and (M DIV DOMAIN <= DomainsPerBlock)
            then
            begin
              SUM := SUM / DOMAIN;

              if (SUM > 1.0)
                then
                  SUM := 1.0;

              TPRIN := (TSTART + (IT - 1) * TSTEP) - 273.16;
              IESS := IESS + 1;
              SUMW[IESS] := SUM;

{ This is where 3D values are stored into MapDataPoint ProDataPoint arrays }
{ Write for 3-D data follows }
              if (TPRIN + TSTEP) <= TF { Haven't passed final temp }
                then
                begin
                  { Store values into MapDataPoint array }
                  MapDataPoint[MapDataPointNumber].Temp := TPRIN + TSTEP;
                  MapDataPoint[MapDataPointNumber].Pos := M - DOMAIN DIV 2;
                  MapDataPoint[MapDataPointNumber].Theta := SUM;

                  { Store values into ProDataPoint array }
                  ProDataPoint[ProDataPointNumber].Pos := PrevBasesRead + MapDataPoint[MapDataPointNumber].Pos;
                  { Store temp only if 0.5 theta is crossed }
                  if (MapDataPoint[MapDataPointNumber - DomainsPerBlock].Theta > 0.5)
                   and (MapDataPoint[MapDataPointNumber].Theta <= 0.5)
                    then
                      { Temp must be interpolated }
                      ProDataPoint[ProDataPointNumber].Temp := MapDataPoint[MapDataPointNumber].Temp
                       - (0.5 - MapDataPoint[MapDataPointNumber].Theta)
                       / (MapDataPoint[MapDataPointNumber - DomainsPerBlock].Theta
                       - MapDataPoint[MapDataPointNumber].Theta) * TSTEP;

                  { Increase ProDataPointNumber }
                  if ProDataPointNumber < DomainsPerBlock
                    then
                      ProDataPointNumber := ProDataPointNumber + 1
                    else
                      ProDataPointNumber := 1;

                  { Increase MapDataPointNumber }
                  if MapDataPointNumber < NumOfMapDataPoints
                    then
                      MapDataPointNumber := MapDataPointNumber + 1;
                end; { if TPRIN + TSTEP...}
{ 2902 }
{ 2903 }
              SUM := 0;
              IESS := 0;
            end; { if M MOD DOMAIN... }
        end; { if IT >= 0... }

{ End of denatuation map section }

{ 3000 }
      THETA := THETA + PM;

    end; { for M := 1 to N1... }

{ Calculate the information for the differential melting curve. }

    THETA := THETA / NMAX;
    DTT := (THETA0 - THETA) / TSTEP;

    THETA0 := THETA;

    if (IT < 0)
      then
        Goto IncreaseIT;

{ For temperature and derivative THETA output: }
{ This is where 2D data points are averaged and stored }

    if MltDataPointNumber <= NumOfMltDataPoints
      then
      begin
        { Find the highest or last mltdatapoint for plots regardless of block melts }
        if MltDataPointNumber > LastMltDataPoint
          then
            LastMltDataPoint := MltDataPointNumber;

        { Enter 2D BLOCK melting curve data points into BlockMltDataPoint array }
        if BlockMltDataPoint[MltDataPointNumber].Temp = 0 { Record temps once }
          then
            BlockMltDataPoint[MltDataPointNumber].Temp := TEMP - 273.16;
        BlockMltDataPoint[MltDataPointNumber].DthetaDT := DTT;

        { Enter 2D TOTAL melting curve data points into TotalMltDataPoint array }
        if TotalMltDataPoint[MltDataPointNumber].Temp = 0 { Record temps once }
          then
            TotalMltDataPoint[MltDataPointNumber].Temp := TEMP - 273.16;
        TotalMltDataPoint[MltDataPointNumber].DthetaDT :=
         ((TotalMltDataPoint[MltDataPointNumber].DthetaDT * (TotalBasesMelted - BlockLENGTH)) + (DTT * BlockLENGTH))
          / TotalBasesMelted;

        MltDataPointNumber := MltDataPointNumber + 1;
      end; { if }

    { Convert temp, block, and total to strings for output }
    Str((TEMP - 273.16):1:2, TempString);
    Str(BlockLENGTH, BlockLENGTHstring);
    Str(TotalBasesMelted:1:0, TotalBasesMeltedString);
    TempString := TempString + 'C';
    BlockLENGTHstring := BlockLENGTHstring + 'bp';
    TotalBasesMeltedString := TotalBasesMeltedString + 'bp';


{ 4005 }
{
FOR COMPACT 6(E10.3) OUTPUT OF DTT
      IW := 1 + (IT MOD 6);
      THETW(IW) := DTT;
}
{ 4000 }
{ 4010 }

    if IT < (NITS - 1)
      then
        Goto IncreaseIT;

    { Block finished melting }
    { Ouput profile--OutFile3 }
    { File is appended each time }
{Writeln(OutFile3, '*');
    for X := 1 to DomainsPerBlock do
      Writeln(OutFile3, ProDataPoint[X].Pos:1:0, Chr(9), ProDataPoint[X].Temp:8:4);}
  end; { MeltAblock }


  (**************************************************************************)


begin { MeltAfile }


  { Open all output files }
   writeln(InFileName,' ', OutFile1Name,' ', OutFile2Name,' ',LogFileName );
  assign(InFile, InFileName);
  Reset(InFile);
  assign(OutFile1, OutFile1Name);
  Rewrite(OutFile1);
  assign(OutFile2, OutFile2Name);
  Rewrite(OutFile2);
  assign(LogFile, LogFileName);
  Rewrite(LogFile);

  { Begin writing log }
  Writeln(LogFile, 'Program                   : MeltSim');
  Writeln(LogFile, 'Sequence File             : ', InFileName);
  Writeln(LogFile, 'Title                     : ', 'Melting Curve');
  Writeln(LogFile, 'Curve Output File         : ', OutFile1Name);
  Writeln(LogFile, 'Sodium Ion Concentration  : ', SALT:1:3, ' M');
  Writeln(LogFile, 'Temperature Increment     : ', TSTEP:1:2, ' C');
  Writeln(LogFile, 'Starting Temperature      : ', TS:1:0, ' C');
  Writeln(LogFile, 'Final Temperature         : ', TF:1:0, ' C');

  { Find the exact number of mltdatapoints used }
  NumOfMltDataPoints := Round((TF - TS) / TSTEP) + 1;
  if NumOfMltDataPoints > MaxMltDataPoints
    then
      NumOfMltDataPoints := MaxMltDataPoints;

  { Find the number of mapdatapoints used }
  DomainsPerBlock := 60; {INPUT}
  NumOfMapDataPoints := NumOfMltDataPoints * DomainsPerBlock;

  { Initialize mltdatapoint arrays }
  for X := 1 to MaxMltDataPoints do
  begin
    TotalMltDataPoint[X].Temp := 0;
    TotalMltDataPoint[X].DthetaDT := 0;
    BlockMltDataPoint[X].Temp := 0;
    BlockMltDataPoint[X].DthetaDT := 0;
  end;

  { Reset variables }
  MltDataPointNumber := 1;
  LastMltDataPoint := 1;
  MapDataPointNumber := 1;
  ProDataPointNumber := 1;
  BlockMltActive := True;
  TotalBasesRead := 0;
  TotalBasesMelted := 0;
  PrevBasesRead := 0;

  { This is the main loop for block melts }
  while not(EOF(InFile)) do
    MeltAblock;

  BlockMltActive := False;

  { Ouput curve - OutFile1 and OutFile2 for the integration }
   MltDataSum :=0;
  for X := 1 to LastMltDataPoint do
  begin
     MltDataSum := MltDataSum + TotalMltDataPoint[X].DthetaDT;
     Writeln(OutFile1, TotalMltDataPoint[X].Temp:4:2, Chr(9),
	     TotalMltDataPoint[X].DthetaDT:3:6);
     Writeln(OutFile2, TotalMltDataPoint[X].Temp:4:2, Chr(9),
	     MltDataSum:3:6);
  end;
  { Finish writing log }
   Writeln(LogFile, 'Domain Size: ', DOMAIN, ' bp');
   Writeln(LogFile, 'Bases Melted: ', TotalBasesMelted:1:0, 'bp (ACGTX)');
      
   close(InFile);
   close(OutFile1);
   close(OutFile2);
   close(LogFile);
   
end; { MeltAfile }


(******************************************************************************)

procedure GetParameters;
var
   par : text;
begin {GetParameters}
   PARAMC := 1.75;
   SALT := 0.075;{ sodium concentration}
   TSTEP := 0.1;{ temperature increment}
   TS := 40; { starting temp.}
   TF := 120; { final temp.}

   assign(par,'melt_param.dat');
   reset(par);
   readln(par);
   readln(par, PARAMC,SIGMAP,TS,TF,TSTEP,SALT);

 end; { GetParameters }

(*****************************************************************************)

procedure GetFileNames;
var
   parnames	  : text;
   i		  : longint;
   name_i, name_o : string;
   tlen		  : integer;
begin {GetFileNames}
   { OutFileNames come from parameters }
   Str(index,strindex);

   assign(parnames,'melt_files.dat');
   reset(parnames);
   readln(parnames,name_i);
   readln(parnames,name_o);

   i:=-1;
   repeat
   begin
      i:=i+1;
      if(name_i[i]<>' ')then  InName[i]:=name_i[i];
   end;
   until (i=length(name_i));
   writeln(InName,'$');

   i:=-1;
   repeat
   begin
      i:=i+1;
      if(name_o[i]<>' ')then  OutBase[i]:=name_o[i];
   end;
   until (i=length(name_o));
   writeln(OutBase,'$');

   { OutFileNames come from parameters }
   OutFile1Name := OutBase+'.mlt'; { EMltFile }
   OutFile2Name := OutBase+'.int'; { EIntFile}
   LogFileName  := OutBase+'.log'; { ELogFile }

end; { GetFileNames }

(*****************************************************************************)

procedure execute_loop;
begin{execute_loop}
   PARAMC := 0.0;
   SALT := 0.075;
   TSTEP := 0.1;
   TS := 40;
   TF := 140;
   SIGMAP:=0.0635;

   (* Configure the loop over the regarded parameter
    * loop can be made over PARAMC, SIGMAP, SALT and TSTEP
    * just change parameter name and increment in followong code
    * change also the basestr for the output files base string
    *)
   basestr :='Datas/c'; (* ! create directory if mentionned in basestr*)
   for index :=0 to 50 do
   begin
      PARAMC := PARAMC+0.1;
      { OutFileNames come from parameters }
      Str(PARAMC:5:5,strindex);
      OutFile1Name := outputdir+basestr+strindex+'.mlt'; { EMltFile }
      OutFile2Name := outputdir+basestr+strindex+'.int'; { Integrate file}
      LogFileName  := outputdir+basestr+strindex+'.log'; { ELogFile }
      writeln(PARAMC); 
      MeltAfile(InName, 1);
   end;
   
end; { execute_loop }

(*****************************************************************************)

procedure execute;
begin{execute}
   index := index+1;

   MeltAfile(InName, 1);
   
end; { execute }



(*****************************************************************************)
(*****************************************************************************)

begin { TMeltThread.Execute }

   index :=0;
   PARAMC := 1.75;
   SIGMAP := 1e-5;
   SALT := 0.075;{ sodium concentration}
   TSTEP := 0.1;{ temperature increment}
   TS := 40; { starting temp.}
   TF := 120; { final temp.}
   outputdir := './';
   OutFile1Name := 'melt.mlt'; { EMltFile }
   OutFile2Name := 'melt.int'; { EIntFile}
   LogFileName  := 'melt.log'; { ELogFile }

   
   GetParameters;
   GetFileNames;(*from file: optionnal : ! carefull put no space in the file*)

   {procedure execute      : reads parameters from a file}
   {procedure execute_loop : loop the code over one parameter}
   execute;

   writeln('Sequence melt.');
end.

